import { of } from "rxjs";
import { ExecutionContext, CallHandler } from "@nestjs/common";
import { Reflector } from "@nestjs/core";
import { ResponseEnvelopeInterceptor } from "./response-envelope.interceptor";
import * as envelope from "@shared/http/response-envelope";
import type { ResponseEnvelope } from "@shared/http/response-envelope";

/**
 * Mock ExecutionContext
 */
function createContext(): ExecutionContext {
  return {
    getHandler: jest.fn(),
    getClass: jest.fn(),
  } as unknown as ExecutionContext;
}

/**
 * Mock CallHandler
 */
function createNext(data: unknown): CallHandler {
  return {
    handle: () => of(data),
  };
}

describe("ResponseEnvelopeInterceptor", () => {
  let interceptor: ResponseEnvelopeInterceptor;
  let reflector: Reflector;

  beforeEach(() => {
    reflector = {
      getAllAndOverride: jest.fn(),
    } as unknown as Reflector;

    interceptor = new ResponseEnvelopeInterceptor(reflector);
  });

  it("should return data directly when skip envelope is true", (done) => {
    jest.spyOn(reflector, "getAllAndOverride").mockReturnValue(true);

    const ctx = createContext();
    const next = createNext({ foo: "bar" });

    interceptor.intercept(ctx, next).subscribe((result) => {
      expect(result).toEqual({ foo: "bar" });
      done();
    });
  });

  it("should wrap data when skip is false and data is not enveloped", (done) => {
    jest.spyOn(reflector, "getAllAndOverride").mockReturnValue(false);
    jest.spyOn(envelope, "isAlreadyEnveloped").mockReturnValue(false);

    const envelopedResult: ResponseEnvelope<unknown> = {
      transactionNo: "TEST-TRANSACTION",
      timestamp: new Date().toISOString(),
      status: true,
      data: { foo: "bar" },
    };

    const toEnvelopeSpy = jest
      .spyOn(envelope, "toEnvelope")
      .mockReturnValue(envelopedResult);

    const ctx = createContext();
    const next = createNext({ foo: "bar" });

    interceptor.intercept(ctx, next).subscribe((result) => {
      expect(toEnvelopeSpy).toHaveBeenCalled();
      expect(result).toEqual(envelopedResult);
      done();
    });
  });

  it("should return data as-is when already enveloped", (done) => {
    jest.spyOn(reflector, "getAllAndOverride").mockReturnValue(false);
    jest.spyOn(envelope, "isAlreadyEnveloped").mockReturnValue(true);

    const data: ResponseEnvelope<unknown> = {
      transactionNo: "TEST-TRANSACTION",
      timestamp: new Date().toISOString(),
      status: true,
      data: { foo: "bar" },
    };

    const ctx = createContext();
    const next = createNext(data);

    interceptor.intercept(ctx, next).subscribe((result) => {
      expect(result).toBe(data);
      done();
    });
  });
});
