export interface ErrorExt {
  code: string;
  message: string;
  description: string;
  path: string;
  type: string;
  technicalError?: string;
}

export function toHttpStatus(code: string): number {
  const first = code.charAt(0);
  if (first === "3") return 401;
  if (first === "4") return 400;
  if (first === "6") return 422;
  return 500;
}

type ErrorTuple = [string, string, string];
type SimpleErrorPair = [string, string];

interface ErrorGroupDefinition {
  prefixes: string[];
  systemNames: string[];
  errors: Record<string, SimpleErrorPair>;
}

function generateErrorMaps(
  defs: ErrorGroupDefinition[],
): Record<string, ErrorTuple> {
  const result: Record<string, ErrorTuple> = {};

  for (const def of defs) {
    for (let i = 0; i < def.prefixes.length; i++) {
      const prefix = def.prefixes[i];
      const systemName = def.systemNames[i];

      for (const [suffix, pair] of Object.entries(def.errors)) {
        const [message, description] = pair;
        result[`${prefix}${suffix}`] = [systemName, message, description];
      }
    }
  }
  return result;
}

const AUTHORIZATION_ERRORS: Record<string, SimpleErrorPair> = {
  "001": ["Unauthorization", "Username or password is incorrect"],
  "002": ["Signature Unauthorization", "Authorization Error"],
  "003": ["Unauthorization", "Invalid token or refresh token"],
  "004": ["Unauthorization", "Authorization Error"],
};

const INVALID_PARAMETER_ERRORS: Record<string, SimpleErrorPair> = {
  "001": ["Data Not Found Error", "Data Not Found Error"],
  "002": ["Invalid Parameter Request", "Invalid Parameter Request"],
  "003": ["Data is Wrong Format", "Data is Wrong Format"],
  "004": ["Data duplicate Error", "Data duplicate Error"],
};

const ERROR_HANDLING_ERRORS: Record<string, SimpleErrorPair> = {
  "001": ["Internal Server Error", "Error Handling Errors"],
  "002": ["Service Connection Error", "Error Handling Errors"],
  "003": ["Service Connection Timeout", "Error Handling Errors"],
  "004": ["Connect DataBase Error", "Error Handling Errors"],
};

const BUSINESS_VALIDATION_ERRORS_61: Record<string, SimpleErrorPair> = {
  "001": ["Chassis is Duplicated", "Business Validation"],
  "002": ["This Customer is Block By AML", "Business Validation"],
  "003": ["Loan validation Error", "Business Validation"],
  "004": ["Sender is Invalid", "Business Validation"],
  "005": ["Signature is Invalid", "Business Validation"],
  "006": ["The recipient sequence is out of sequence", "Business Validation"],
  "007": [
    "The pagenumber specified in the tab element is not in the document",
    "Business Validation",
  ],
  "008": [
    "The recipient you have identified is not a valid recipient",
    "Business Validation",
  ],
};

const BUSINESS_VALIDATION_ERRORS_62: Record<string, SimpleErrorPair> = {
  "001": ["Chassis is Duplicated", "Business Validation"],
  "002": ["This Customer is Block By AML", "Business Validation"],
  "003": ["Loan validation Error", "Business Validation"],
  "004": ["Sender is Invalid", "Business Validation"],
  "005": ["Signature is Invalid", "Business Validation"],
  "006": ["This referenceNo have already registered", "Business Validation"],
  "007": [
    "Your referenceNo or eSignatureNo not registered yet",
    "Business Validation",
  ],
  "008": ["Esignature is processing", "Business Validation"],
};

const BUSINESS_VALIDATION_ERRORS_63: Record<string, SimpleErrorPair> = {
  "001": ["Chassis is Duplicated", "Business Validation"],
  "002": ["This Customer is Block By AML", "Business Validation"],
  "003": ["Loan validation Error", "Business Validation"],
  "004": ["Sender is Invalid", "Business Validation"],
  "005": ["Signature is Invalid", "Business Validation"],
  "006": ["This Order has been Processing", "Business Validation"],
  "007": ["This referenceNo have already registered", "Business Validation"],
  "008": [
    "Your referenceNo or eSignatureNo not registered yet",
    "Business Validation",
  ],
  "009": ["Sign position not match with document", "Business Validation"],
  "010": ["Your document is not a pdf file", "Business Validation"],
  "011": [
    "Comfirm document not match in document registed",
    "Business Validation",
  ],
  "012": ["Esignature is processing", "Business Validation"],
  "013": [
    "Not able to get envelope from an envelope that has not created",
    "Business Validation",
  ],
  "014": ["Your signer sequence is out of sequence", "Business Validation"],
  "015": ["The envelope was declined", "Business Validation"],
  "016": [
    "The pagenumber specified in the tab element is not in the document",
    "Business Validation",
  ],
  "017": [
    "The signer you have identified is not a valid signer",
    "Business Validation",
  ],
};

const BUSINESS_VALIDATION_ERRORS_64: Record<string, SimpleErrorPair> = {
  "001": ["Chassis is Duplicated", "Business Validation"],
  "002": ["This Customer is Block By AML", "Business Validation"],
  "003": ["Loan validation Error", "Business Validation"],
  "004": ["Sender is Invalid", "Business Validation"],
  "005": ["Signature is Invalid", "Business Validation"],
  "006": ["Service not avalible", "Business Validation"],
};

const SQL_ERRORS: Record<string, SimpleErrorPair> = {
  "001": ["Insert SQL Error", "SQL Error"],
  "002": ["Update SQL Error", "SQL Error"],
  "003": ["Select SQL Error", "SQL Error"],
  "004": ["Update State SQL Error", "SQL Error"],
};

const ERROR_DEFINITIONS: ErrorGroupDefinition[] = [
  {
    prefixes: ["31", "32", "33", "34", "35"],
    systemNames: [
      "Authorization Error | System I",
      "Authorization Error | System II",
      "Authorization Error | Process",
      "Authorization Error | UX",
      "Authorization Error | BFF",
    ],
    errors: AUTHORIZATION_ERRORS,
  },
  {
    prefixes: ["41", "42", "43", "44", "45"],
    systemNames: [
      "Invalid Parameter Request | System I",
      "Invalid Parameter Request | System II",
      "Invalid Parameter Request | Process",
      "Invalid Parameter Request | UX",
      "Invalid Parameter Request | BFF",
    ],
    errors: INVALID_PARAMETER_ERRORS,
  },
  {
    prefixes: ["51", "52", "53", "54", "55"],
    systemNames: [
      "Error Handling Errors | System I",
      "Error Handling Errors | System II",
      "Error Handling Errors | Process",
      "Error Handling Errors | UX",
      "Error Handling Errors | BFF",
    ],
    errors: ERROR_HANDLING_ERRORS,
  },
  {
    prefixes: ["61"],
    systemNames: ["Business Validation | System I"],
    errors: BUSINESS_VALIDATION_ERRORS_61,
  },
  {
    prefixes: ["62"],
    systemNames: ["Business Validation | System II"],
    errors: BUSINESS_VALIDATION_ERRORS_62,
  },
  {
    prefixes: ["63"],
    systemNames: ["Business Validation | Process"],
    errors: BUSINESS_VALIDATION_ERRORS_63,
  },
  {
    prefixes: ["64"],
    systemNames: ["Business Validation | UX"],
    errors: BUSINESS_VALIDATION_ERRORS_64,
  },
  {
    prefixes: ["71", "72"],
    systemNames: ["SQL Error | System I", "SQL Error | System II"],
    errors: SQL_ERRORS,
  },
];

const ERROR_CODE_MAP: Record<string, ErrorTuple> = Object.freeze(
  generateErrorMaps(ERROR_DEFINITIONS),
);

export function newErrorExt(
  prefix: string,
  code: string,
  path: string,
  technicalError = "",
): ErrorExt {
  const mapping = ERROR_CODE_MAP[code];
  if (!mapping) {
    throw new Error("not found error code");
  }
  const [type, message, description] = mapping;
  return {
    code: `${prefix}${code}`,
    message,
    description,
    path,
    type,
    technicalError,
  };
}
