import "reflect-metadata";
import { HttpException, ArgumentsHost } from "@nestjs/common";
import type { Request, Response } from "express";
import { AllExceptionsFilter } from "./all-exceptions.filter";
import { AppError } from "@shared/errors/app-error";

jest.mock("../../../config/env", () => ({
  AppEnv: {
    ERROR_PREFIX: "HRR",
  },
}));

/**
 * Mock express response with separated jest.fn()
 * to avoid @typescript-eslint/unbound-method
 */
type MockResponse = {
  res: jest.Mocked<Response>;
  statusMock: jest.Mock;
  jsonMock: jest.Mock;
};

function createMockResponse(): MockResponse {
  const statusMock = jest.fn();
  const jsonMock = jest.fn();

  const res = {
    status: statusMock,
    json: jsonMock,
  } as unknown as jest.Mocked<Response>;

  statusMock.mockReturnValue(res);
  jsonMock.mockReturnValue(res);

  return { res, statusMock, jsonMock };
}

/**
 * Mock ArgumentsHost
 */
function createMockArgumentsHost(
  req: Partial<Request>,
  res: Response,
): ArgumentsHost {
  return {
    switchToHttp: () => ({
      getRequest: () => req as Request,
      getResponse: () => res,
    }),
  } as ArgumentsHost;
}

describe("AllExceptionsFilter", () => {
  const path = "/test-path";
  let filter: AllExceptionsFilter;

  beforeEach(() => {
    filter = new AllExceptionsFilter("HRR");
  });

  describe("AppError handling", () => {
    it("should handle AppError with correct status and envelope", () => {
      const error = new AppError("41001", path, "not found");
      const req: Partial<Request> = { originalUrl: path };

      const { res, statusMock, jsonMock } = createMockResponse();
      const host = createMockArgumentsHost(req, res);

      filter.catch(error, host);

      expect(statusMock).toHaveBeenCalledWith(error.httpStatus);
      expect(jsonMock).toHaveBeenCalled();
      const calls = jsonMock.mock.calls as Array<
        Array<Record<string, unknown>>
      >;
      const envelope = calls[0]?.[0];
      expect(envelope).toHaveProperty("status", false);
      expect(envelope).toHaveProperty("data");
    });
  });

  describe("HttpException handling", () => {
    it("should handle HttpException with status 400", () => {
      const exception = new HttpException("bad request", 400);
      const req: Partial<Request> = { originalUrl: path };

      const { res, statusMock, jsonMock } = createMockResponse();
      const host = createMockArgumentsHost(req, res);

      filter.catch(exception, host);

      expect(statusMock).toHaveBeenCalledWith(400);
      expect(jsonMock).toHaveBeenCalled();
      const calls = jsonMock.mock.calls as Array<
        Array<Record<string, unknown>>
      >;
      const envelope = calls[0]?.[0];
      expect(envelope).toHaveProperty("status", false);
    });

    it("should handle HttpException with status 404", () => {
      const exception = new HttpException("not found", 404);
      const req: Partial<Request> = { originalUrl: path };

      const { res, statusMock, jsonMock } = createMockResponse();
      const host = createMockArgumentsHost(req, res);

      filter.catch(exception, host);

      expect(statusMock).toHaveBeenCalledWith(404);
      expect(jsonMock).toHaveBeenCalled();
    });

    it("should handle HttpException with object response", () => {
      const responseObj = { message: "validation failed", errors: ["field1"] };
      const exception = new HttpException(responseObj, 400);
      const req: Partial<Request> = { originalUrl: path };

      const { res, statusMock, jsonMock } = createMockResponse();
      const host = createMockArgumentsHost(req, res);

      filter.catch(exception, host);

      expect(statusMock).toHaveBeenCalledWith(400);
      expect(jsonMock).toHaveBeenCalled();
    });

    it("should handle HttpException with status 500", () => {
      const exception = new HttpException("internal server error", 500);
      const req: Partial<Request> = { originalUrl: path };

      const { res, statusMock, jsonMock } = createMockResponse();
      const host = createMockArgumentsHost(req, res);

      filter.catch(exception, host);

      expect(statusMock).toHaveBeenCalledWith(500);
      expect(jsonMock).toHaveBeenCalled();
    });
  });

  describe("Unknown error handling", () => {
    it("should handle Error instance", () => {
      const exception = new Error("unexpected error");
      const req: Partial<Request> = { originalUrl: path };

      const { res, statusMock, jsonMock } = createMockResponse();
      const host = createMockArgumentsHost(req, res);

      filter.catch(exception, host);

      expect(statusMock).toHaveBeenCalledWith(500);
      expect(jsonMock).toHaveBeenCalled();
    });

    it("should handle error with P2002 code (unique constraint)", () => {
      const exception = { code: "P2002", message: "Unique constraint failed" };
      const req: Partial<Request> = { originalUrl: path };

      const { res, statusMock, jsonMock } = createMockResponse();
      const host = createMockArgumentsHost(req, res);

      filter.catch(exception, host);

      expect(statusMock).toHaveBeenCalledWith(400);
      expect(jsonMock).toHaveBeenCalled();
    });

    it("should handle error with P2025 code (record not found)", () => {
      const exception = { code: "P2025", message: "Record not found" };
      const req: Partial<Request> = { originalUrl: path };

      const { res, statusMock, jsonMock } = createMockResponse();
      const host = createMockArgumentsHost(req, res);

      filter.catch(exception, host);

      expect(statusMock).toHaveBeenCalledWith(400);
      expect(jsonMock).toHaveBeenCalled();
    });

    it("should handle error with unknown code", () => {
      const exception = { code: "UNKNOWN", message: "Unknown error" };
      const req: Partial<Request> = { originalUrl: path };

      const { res, statusMock, jsonMock } = createMockResponse();
      const host = createMockArgumentsHost(req, res);

      filter.catch(exception, host);

      expect(statusMock).toHaveBeenCalledWith(500);
      expect(jsonMock).toHaveBeenCalled();
    });

    it("should handle non-Error object without code", () => {
      const exception = { someProperty: "value" };
      const req: Partial<Request> = { originalUrl: path };

      const { res, statusMock, jsonMock } = createMockResponse();
      const host = createMockArgumentsHost(req, res);

      filter.catch(exception, host);

      expect(statusMock).toHaveBeenCalledWith(500);
      expect(jsonMock).toHaveBeenCalled();
    });

    it("should handle string exception", () => {
      const exception = "string error";
      const req: Partial<Request> = { originalUrl: path };

      const { res, statusMock, jsonMock } = createMockResponse();
      const host = createMockArgumentsHost(req, res);

      filter.catch(exception, host);

      expect(statusMock).toHaveBeenCalledWith(500);
      expect(jsonMock).toHaveBeenCalled();
    });

    it("should handle number exception", () => {
      const exception = 12345;
      const req: Partial<Request> = { originalUrl: path };

      const { res, statusMock, jsonMock } = createMockResponse();
      const host = createMockArgumentsHost(req, res);

      filter.catch(exception, host);

      expect(statusMock).toHaveBeenCalledWith(500);
      expect(jsonMock).toHaveBeenCalled();
    });

    it("should handle error object with code and message (not Error instance)", () => {
      const exception = {
        code: "CUSTOM_CODE",
        message: "Custom error message",
      };
      const req: Partial<Request> = { originalUrl: path };

      const { res, statusMock, jsonMock } = createMockResponse();
      const host = createMockArgumentsHost(req, res);

      filter.catch(exception, host);

      expect(statusMock).toHaveBeenCalledWith(500);
      expect(jsonMock).toHaveBeenCalled();
    });

    it("should handle P2002 error object with message (not Error instance)", () => {
      const exception = { code: "P2002", message: "Unique constraint error" };
      const req: Partial<Request> = { originalUrl: path };

      const { res, statusMock, jsonMock } = createMockResponse();
      const host = createMockArgumentsHost(req, res);

      filter.catch(exception, host);

      expect(statusMock).toHaveBeenCalledWith(400);
      expect(jsonMock).toHaveBeenCalled();
    });

    it("should handle P2025 error object with message (not Error instance)", () => {
      const exception = { code: "P2025", message: "Record not found error" };
      const req: Partial<Request> = { originalUrl: path };

      const { res, statusMock, jsonMock } = createMockResponse();
      const host = createMockArgumentsHost(req, res);

      filter.catch(exception, host);

      expect(statusMock).toHaveBeenCalledWith(400);
      expect(jsonMock).toHaveBeenCalled();
    });

    it("should handle error object with numeric code (not string)", () => {
      const exception = { code: 12345, message: "Error with numeric code" };
      const req: Partial<Request> = { originalUrl: path };

      const { res, statusMock, jsonMock } = createMockResponse();
      const host = createMockArgumentsHost(req, res);

      filter.catch(exception, host);

      expect(statusMock).toHaveBeenCalledWith(500);
      expect(jsonMock).toHaveBeenCalled();
    });

    it("should handle null exception", () => {
      const exception = null;
      const req: Partial<Request> = { originalUrl: path };

      const { res, statusMock, jsonMock } = createMockResponse();
      const host = createMockArgumentsHost(req, res);

      filter.catch(exception, host);

      expect(statusMock).toHaveBeenCalledWith(500);
      expect(jsonMock).toHaveBeenCalled();
    });

    it("should handle undefined exception", () => {
      const exception = undefined;
      const req: Partial<Request> = { originalUrl: path };

      const { res, statusMock, jsonMock } = createMockResponse();
      const host = createMockArgumentsHost(req, res);

      filter.catch(exception, host);

      expect(statusMock).toHaveBeenCalledWith(500);
      expect(jsonMock).toHaveBeenCalled();
    });

    it("should handle boolean exception", () => {
      const exception = false;
      const req: Partial<Request> = { originalUrl: path };

      const { res, statusMock, jsonMock } = createMockResponse();
      const host = createMockArgumentsHost(req, res);

      filter.catch(exception, host);

      expect(statusMock).toHaveBeenCalledWith(500);
      expect(jsonMock).toHaveBeenCalled();
    });
  });

  describe("Request path handling", () => {
    it("should use req.url when originalUrl is not available", () => {
      const exception = new Error("test error");
      const req: Partial<Request> = { url: "/backup-path" };

      const { res, statusMock, jsonMock } = createMockResponse();
      const host = createMockArgumentsHost(req, res);

      filter.catch(exception, host);

      expect(statusMock).toHaveBeenCalled();
      expect(jsonMock).toHaveBeenCalled();
    });

    it("should use empty string when both originalUrl and url are not available", () => {
      const exception = new Error("test error");
      const req: Partial<Request> = {};

      const { res, statusMock, jsonMock } = createMockResponse();
      const host = createMockArgumentsHost(req, res);

      filter.catch(exception, host);

      expect(statusMock).toHaveBeenCalled();
      expect(jsonMock).toHaveBeenCalled();
    });
  });

  describe("Constructor", () => {
    it("should create filter with default error prefix", () => {
      const defaultFilter = new AllExceptionsFilter();
      expect(defaultFilter).toBeDefined();
    });

    it("should create filter with custom error prefix", () => {
      const customFilter = new AllExceptionsFilter("CUSTOM");
      expect(customFilter).toBeDefined();

      const exception = new Error("test");
      const req: Partial<Request> = { originalUrl: path };
      const { res, statusMock, jsonMock } = createMockResponse();
      const host = createMockArgumentsHost(req, res);

      customFilter.catch(exception, host);

      expect(statusMock).toHaveBeenCalled();
      expect(jsonMock).toHaveBeenCalled();
    });

    it("should use HRR as fallback when ERROR_PREFIX is undefined", () => {
      // Save original value
      // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment, @typescript-eslint/no-unsafe-member-access
      const { AppEnv } = jest.requireMock("../../../config/env");
      // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment, @typescript-eslint/no-unsafe-member-access
      const originalPrefix = AppEnv.ERROR_PREFIX;

      // Mock ERROR_PREFIX as undefined
      // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment
      Object.defineProperty(AppEnv, "ERROR_PREFIX", {
        get: () => undefined,
        configurable: true,
      });

      const filterWithFallback = new AllExceptionsFilter();
      expect(filterWithFallback).toBeDefined();

      const exception = new Error("test");
      const req: Partial<Request> = { originalUrl: path };
      const { res, statusMock, jsonMock } = createMockResponse();
      const host = createMockArgumentsHost(req, res);

      filterWithFallback.catch(exception, host);

      expect(statusMock).toHaveBeenCalled();
      expect(jsonMock).toHaveBeenCalled();

      // Restore original value
      // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment, @typescript-eslint/no-unsafe-member-access
      Object.defineProperty(AppEnv, "ERROR_PREFIX", {
        // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment
        value: originalPrefix,
        configurable: true,
      });
    });

    it("should use HRR as fallback when ERROR_PREFIX is null", () => {
      // Save original value
      // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment, @typescript-eslint/no-unsafe-member-access
      const { AppEnv } = jest.requireMock("../../../config/env");
      // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment, @typescript-eslint/no-unsafe-member-access
      const originalPrefix = AppEnv.ERROR_PREFIX;

      // Mock ERROR_PREFIX as null
      // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment
      Object.defineProperty(AppEnv, "ERROR_PREFIX", {
        get: () => null,
        configurable: true,
      });

      const filterWithNull = new AllExceptionsFilter();
      expect(filterWithNull).toBeDefined();

      const exception = new Error("test");
      const req: Partial<Request> = { originalUrl: path };
      const { res, statusMock, jsonMock } = createMockResponse();
      const host = createMockArgumentsHost(req, res);

      filterWithNull.catch(exception, host);

      expect(statusMock).toHaveBeenCalled();
      expect(jsonMock).toHaveBeenCalled();

      // Restore original value
      // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment, @typescript-eslint/no-unsafe-member-access
      Object.defineProperty(AppEnv, "ERROR_PREFIX", {
        // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment
        value: originalPrefix,
        configurable: true,
      });
    });
  });
});
