import {
  ArgumentsHost,
  Catch,
  ExceptionFilter,
  HttpException,
} from "@nestjs/common";
import { Request, Response } from "express";
import { toEnvelope } from "@shared/http/response-envelope";
import { AppError } from "@shared/errors/app-error";
import { newErrorExt, toHttpStatus } from "@shared/errors/error-ext";
import { AppEnv } from "../../../config/env";

interface ErrorWithCode {
  code?: string;
  message?: string;
}

function hasCode(err: unknown): err is ErrorWithCode {
  return (
    typeof err === "object" &&
    err !== null &&
    "code" in err &&
    typeof (err as ErrorWithCode).code === "string"
  );
}

function mapUnknownToCode(err: unknown): string | null {
  if (!hasCode(err)) return null;

  // Example mapping; adjust as needed
  if (err.code === "P2002") return "41004";
  if (err.code === "P2025") return "41001";
  return null;
}

@Catch()
export class AllExceptionsFilter implements ExceptionFilter {
  constructor(
    private readonly errorPrefix: string = AppEnv.ERROR_PREFIX ?? "HRR",
  ) {}

  catch(exception: unknown, host: ArgumentsHost) {
    const ctx = host.switchToHttp();
    const req = ctx.getRequest<Request>();
    const res = ctx.getResponse<Response>();
    const path = req.originalUrl ?? req.url ?? "";

    if (exception instanceof AppError) {
      const errorExt = exception.toErrorExt(this.errorPrefix);
      return res.status(exception.httpStatus).json(toEnvelope(errorExt, false));
    }

    if (exception instanceof HttpException) {
      const status = exception.getStatus();
      const resp = exception.getResponse();
      const technical = typeof resp === "string" ? resp : JSON.stringify(resp);
      let code = "51001";
      if (status === 400) code = "41003";
      else if (status === 404) code = "41001";

      const errorExt = newErrorExt(this.errorPrefix, code, path, technical);
      return res.status(status).json(toEnvelope(errorExt, false));
    }

    const mapped = mapUnknownToCode(exception);
    const code = mapped ?? "51001";

    let technical = "";
    if (exception instanceof Error) {
      technical = exception.message;
    } else if (hasCode(exception) && exception.message) {
      technical = exception.message;
    } else {
      technical = String(exception);
    }

    const status = toHttpStatus(code);
    const errorExt = newErrorExt(this.errorPrefix, code, path, technical);
    return res.status(status).json(toEnvelope(errorExt, false));
  }
}
